//
// (c) 2023 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12
import QtQml   2.12

import wesual.Ui     1.0
import wesual.Create 1.0

import wesual.data.Core 1.0

Item {
    id : deleter

    property list <Document> documents
    signal deleted()

    function deleteDocuments() {
        p_.deleteDocuments();
    }

    function isValidFolder(document) {
        switch (document.type) {
        case "areafolder":
        case "spreadsheetfolder":
            return true;
        default:
            return false;
        }
    }

    QtObject {
        id : p_

        property QtObject currentRequest

        function addFolder(list, folder) {
            list.push(folder);

            for (var i = 0; i < folder.childFolders.length; ++i) {
                addFolder(list, folder.childFolders[i]);
            }

            if (folder.type === "areafolder") {
                for (i = 0; i < folder.areas.length; ++i) {
                    addDocument(list, folder.areas[i]);
                }
            } else if (folder.type === "spreadsheetfolder") {
                for (i = 0; i < folder.spreadSheets.length; ++i) {
                    addDocument(list, folder.spreadSheets[i]);
                }
            }
        }

        function addDocument(list, area) {
            list.push(area);
        }

        function deleteDocuments() {
            if (currentRequest) {
                currentRequest.destroy();
            }

            var list = [];

            for (var i = 0; i < deleter.documents.length; ++i) {
                var doc = documents[i];
                if (isValidFolder(doc)) {
                    addFolder(list, doc);
                } else {
                    addDocument(list, doc);
                }
            }

            currentRequest = deleteRequestComponent.createObject(
                        deleter, { documents : list });
            if (list.length > 10) {
                progressDialog.show();
            }
        }
    }

    UiDialog {
        id : progressDialog

        label : qsTrId("e931b324759302ca")

        onRejected : {
            if (p_.currentRequest)
                p_.currentRequest.destroy();
        }

        contentDelegate : Item {
            implicitWidth : 400
            implicitHeight : 42

            UiProgressBar {
                progress : p_.currentRequest ? p_.currentRequest.checkProgress :
                                               0

                anchors {
                    left : parent.left
                    right : parent.right
                    margins : 15
                    verticalCenter : parent.verticalCenter
                }
            }
        }

        buttons : UiDialogButton {
            role : UiDialogButton.CancelRole
            style : Ui.PrimaryPushButtonStyle
        }
    }

    Component {
        id : deleterComponent

        DocumentDeleter {}
    }

    DeleteReferencedDialog {
        id : deleteReferencedDialog

        currentRequest : p_.currentRequest

        onRejected : {
            if (p_.currentRequest)
                p_.currentRequest.destroy();
        }

        onAccepted : {
            if (p_.currentRequest)
                p_.currentRequest.destroy();
        }
    }

    Component {
        id : deleteRequestComponent

        QtObject {
            id : deleteRequest

            property var folders : []
            property var all : []
            property var unreferenced : []
            property var referenced : []
            property var locked : []
            property var unchecked : []

            property list<Document> documents
            property list<DocumentDeleter> deleters

            property int ready : 0

            property real checkProgress : ready / Math.max(1, deleters.length)

            function isReady(status) {
                if (status === DocumentDeleter.Loaded) {
                    ready++;
                    if (ready === deleters.length) {
                        progressDialog.accept(null);

                        doDelete();
                    }
                }
            }

            function checkboxFunctions(deleter, command) {
                var blacklist = unchecked;
                switch (command) {
                case "add" :
                    blacklist.push(deleter);
                    unchecked = blacklist;
                    break;
                case "remove" :
                    let idx = blacklist.indexOf(deleter);
                    if (idx > -1) {
                        blacklist.splice(idx, 1);
                    }
                    unchecked = blacklist
                    break;
                case "find" :
                    return unchecked.indexOf(deleter);
                default:
                    console.error("invalid command " + command);
                    return;
                }
            }

            function addToBlacklist(list, blacklist) {
                for (var i = 0; i < list.length; ++i) {
                    var folder = list[i].document.parentFolder;
                    while (folder) {
                        blacklist[folder.documentId] = true;
                        folder = folder.parentFolder;
                    }
                }
            }

            function deleteFolders(blacklist) {
                for (var i = 0; i < folders.length; ++i) {
                    var folder = folders[i].document;
                    if (!blacklist[folder.documentId]) {
                        folders[i].deleteDocument();
                    }
                }
            }

            function deleteAll() {
                Creator.context.beginRecordChanges();

                for (var i = 0; i < all.length; ++i) {
                    if (checkboxFunctions(all[i], "find") < 0) {
                        all[i].deleteDocument();
                    }
                }

                var blacklist = {};
                addToBlacklist(locked, blacklist);
                addToBlacklist(unchecked, blacklist);
                deleteFolders(blacklist);

                Creator.context.endRecordChanges();
                deleted();
            }

            function deleteUnreferenced() {

                Creator.context.beginRecordChanges();
                for (var i = 0; i < unreferenced.length; ++i) {
                    unreferenced[i].deleteDocument();
                }

                var blacklist = {};
                addToBlacklist(locked, blacklist);
                addToBlacklist(referenced, blacklist);
                deleteFolders(blacklist);

                Creator.context.endRecordChanges();

                deleted();
            }

            function doDelete() {
                let allModified = false;
                let refModified = false;
                let unRefModified = false;
                let foldersModified  = false;
                let lockedModified  = false;

                for (var i = 0; i < deleters.length; ++i) {
                    var deleter = deleters[i];
                    var doc = deleter.document;

                    if (isValidFolder(doc)) {
                        folders.push(deleter);
                        foldersModified = true;
                    } else {
                        if (!deleter.locked) {
                            all.push(deleter);
                            allModified = true;
                            if (deleter.referenced) {
                                referenced.push(deleter);
                                refModified = true;
                            } else {
                                unreferenced.push(deleter);
                                unRefModified = true;
                            }
                        } else {
                            locked.push(deleter);
                            lockedModified = true;
                        }
                    }
                }

                if (allModified)
                    allChanged();
                if (refModified)
                    referencedChanged();
                if (unRefModified)
                    unreferencedChanged();
                if (foldersModified)
                    foldersChanged();
                if (lockedModified)
                    lockedChanged();

                if (referenced.length > 0 || locked.length > 0) {
                    deleteReferencedDialog.show();
                } else {
                    deleteAll();
                }
            }

            Component.onCompleted : {
                var deleters = [];
                for (var i = 0; i < documents.length; ++i) {
                    var deleter = deleterComponent.createObject(
                                deleteRequest, { document : documents[i] });
                    deleter.onStatusChanged.connect(isReady);
                    deleters.push(deleter);
                }
                deleteRequest.deleters = deleters;
            }
        }
    }
}
